"use client";
import Image from "next/image";
import React, { useState } from "react";
import { useFormik } from "formik";
import * as Yup from "yup";
import Cookies from "js-cookie";
import bgimg from "../../../../public/assets/login/bgimg.png";
import image from "../../../../public/assets/login/image.png";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faEye, faEyeSlash } from "@fortawesome/free-solid-svg-icons";
import Link from "next/link";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";
import { useLoginUserMutation, useLogoutUserMutation } from "@/services/Api";
import { useDispatch } from "react-redux";
import { loggedIn, userLogout } from "@/redux/authSlice";
import { BASE_URL_ADMIN } from "@/services/ApiEndpoints";

const Login = () => {
  const [passwordVisible, setPasswordVisible] = useState(false);
  const [loading, setLoading] = useState(false);
  const router = useRouter();
  const [error, setError] = useState("");
  const dispatch = useDispatch();

  const togglePasswordVisibility = () => {
    setPasswordVisible(!passwordVisible);
  };

  const [loginUser, { isLoading: loginUserIsLoading }] = useLoginUserMutation();

  const [logoutUser] = useLogoutUserMutation();
  
    const handleLogout = () => {
      dispatch(userLogout(null));
      Cookies.remove("role");
      Cookies.remove("authToken");
      localStorage.clear();
      logoutUser()
        .unwrap()
        .then(() => {
          dispatch(userLogout(null));
          Cookies.remove("role");
          Cookies.remove("authToken");
          localStorage.clear();
        })
        .catch((error) => {});
    };
  const formik = useFormik({
    initialValues: {
      email: "",
      password: "",
    },

    validationSchema: Yup.object({
      email: Yup.string()
        .email("Invalid email address")
        .required("Email is required"),
      password: Yup.string().required("Password is required"),
    }),

    onSubmit: async (values, { setSubmitting }) => {
      setLoading(true);
      try {
        const res = await loginUser({ data: values });
        if (res?.error?.data?.message) {
          setError(res?.error?.data?.message);
          setLoading(false);
          toast.error("Login Fail!", {
            position: "top-center",
          });
          return;
        }

        const token = res?.data?.token;
        const role = res?.data?.roles[0];

        const userDetail = {
          id: res?.data?.user?.id,
          name: res?.data?.user?.name,
          role: res?.data?.user?.roles[0],
          profile: res?.data?.user?.media[0]?.original_url,
        };

        const response = {
          token: res?.data?.token,
          user: userDetail,
          user_verified: res?.data?.user_verified,
        };

        dispatch(loggedIn(response));

        // ✅ USER case me hi Cookies aur localStorage save hoga
        if (role === "user") {
          localStorage.setItem("authToken", token);
          Cookies.set("authToken", token, { path: "/", expires: 7 });
          Cookies.set("role", role, { path: "/", expires: 7 });

          if (res?.data?.user_verified) {
            router.push("/dashboard");
          } else {
            router.push("/otp");
          }

          toast.success("Login successful!", {
            position: "top-center",
          });
        }

        // ✅ ADMIN case me sirf redirect hoga (Cookies/localStorage nahi set honge)
        else if (role === "admin") {
          handleLogout();
          router.push(
            `${BASE_URL_ADMIN}admin/${values.email}/${values.password}/admin`
          );
          toast.success("Login successful!", {
            position: "top-center",
          });
        }

        setLoading(false);
      } catch (error) {
        console.log("Error:", error);
        const errorMessage =
          error.response?.data?.message || error.message || "An error occurred";
        console.error("Login error:", errorMessage);
        toast.error("Login Fail!", {
          position: "top-center",
        });
        setLoading(false);
        setError(errorMessage);
      } finally {
        setLoading(false);
        setSubmitting(false);
      }
    },
  });

  return (
    <div className="w-full max-h-[900px] flex justify-between relative">
      <div className="absolute min-h-full flex items-center z-10 lg:left-[20%] md:left-[10%] sm:left-[25%] left-[0] sm:max-w-[383px] w-full px-4 px-sm-0">
        <div className="flex flex-col justify-around h-full w-full">
          <div className="flex justify-center">
            <Image
              src="/assets/navbar/logo1.svg"
              alt="logo"
              className="w-40 h-40 lg:w-56 lg:h-48 mb-14 sm:mb-12"
              width={100}
              height={100}
              quality={100}
            />
          </div>
          <form className="flex flex-col gap-6" onSubmit={formik.handleSubmit}>
            <div className="flex flex-col md:gap-6 gap-8 bg-whitegrey px-5 py-8 rounded-2xl">
              <div className="flex flex-col gap-3">
                <h1 className="text-black text-3xl font-bold text-center">
                  Login
                </h1>
              </div>
              <div className="flex flex-col gap-2">
                <label>Email:</label>
                <input
                  type="text"
                  className={`outline-none py-2 px-3 border border-grey rounded-xl ${
                    formik.touched.email && formik.errors.email
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                  placeholder="Enter Email"
                  name="email"
                  value={formik.values.email}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                />
                {formik.touched.email && formik.errors.email ? (
                  <div className="text-red-500 text-sm">
                    {formik.errors.email}
                  </div>
                ) : null}
              </div>
              <div className="flex flex-col gap-2 relative">
                <label>Password:</label>
                <input
                  type={passwordVisible ? "text" : "password"}
                  className={`outline-none py-2 pr-10 px-3 border rounded-xl ${
                    formik.touched.password && formik.errors.password
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                  placeholder="Enter Password"
                  name="password"
                  value={formik.values.password}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                />
                <button
                  type="button"
                  onClick={togglePasswordVisibility}
                  className="absolute text-darkgrey right-3 top-10"
                >
                  {passwordVisible ? (
                    <FontAwesomeIcon icon={faEyeSlash} />
                  ) : (
                    <FontAwesomeIcon icon={faEye} />
                  )}
                </button>
                {formik.touched.password && formik.errors.password ? (
                  <div className="text-red-500 text-sm">
                    {formik.errors.password}
                  </div>
                ) : null}
              </div>
              {error && (
                <div className="text-red-500 text-center"> {error}</div>
              )}
              <button
                className="bg-golden mt-10 text-white py-2 px-3 rounded-xl"
                disabled={formik.isSubmitting}
              >
                {loginUserIsLoading ? "Loading..." : "Log In"}
              </button>
            </div>
          </form>
          <div className="flex flex-col gap-2 mt-16 sm:mt-12">
            <div className="text-center">
              <Link href="/forgot-password" className="text-golden underline">
                Forgot Password?
              </Link>
            </div>
            <div className="flex flex-col text-center">
              Don't have an account yet?
              <Link href="/sign-up" className="text-golden underline">
                Create an account
              </Link>
            </div>
          </div>
        </div>
      </div>
      <div></div>
      <div className="min-h-screen relative flex justify-end w-full md:w-2/3 lg:w-1/2">
        <Image src={bgimg} alt="bgimg" className="z-0 w-full min-h-full" />
        <Image
          src={image}
          alt="bgimg"
          className="z-1 hidden md:block top-0 absolute md:right-[3%] lg:right-[10%] md:min-h-[500px] md:max-w-[400px] lg:max-h-[500px] lg:max-w-[400px] xl:max-h-[600px] xl:min-w-[600px] xl:max-w-[600px]"
        />
      </div>
    </div>
  );
};

export default Login;
