"use client";
import React, { useEffect, useState } from "react";
import studentProfile from "../../../../../public/assets/dashboard/profile.png";
import certificate1 from "../../../../../public/assets/dashboard/certification/image1.png";
import certificateErrorImg from "../../../../../public/assets/CERTIFICATE-2nd-variation.png";
import Image from "next/image";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faDownload, faArrowLeftLong, faArrowRightLong } from "@fortawesome/free-solid-svg-icons";
import Slider from "react-slick";
import "slick-carousel/slick/slick.css";
import "slick-carousel/slick/slick-theme.css";
import Link from "next/link";
import Header from "../header/header";
import { useSelector, useDispatch } from "react-redux";
import { fetchEnrolledCourses } from "@/newStore/features/enrolledCoursesSlice";
import LoaderCircle from "../../loaderCircle/loaderCircle";
import { useAddCertificateMutation, useGetCertificatesQuery } from "@/services/Api";
import toast from "react-hot-toast";
import html2canvas from "html2canvas";
import * as ReactDOM from "react-dom/client";
import CertificateComponent from "../../pdfLayout/PdfLayout";

const Certifications = () => {
  const dispatch = useDispatch();
  const enrolledCourses = useSelector((state) => state.enrolledCourses.enrolledCourses);
  const enrolledCoursesLoader = useSelector((state) => state.enrolledCourses.isLoading);
  const token = useSelector((state) => state.auth.token);
  const { data, isLoading: isCertificatesLoading, refetch } = useGetCertificatesQuery();
  const certificates = data?.certificates || [];
  const [addCertificate, { isLoading: isAddingCertificate }] = useAddCertificateMutation();

  const [loading, setLoading] = useState(false);
  const [generatedCerts, setGeneratedCerts] = useState([]);
  const [hasGenerated, setHasGenerated] = useState(false);

  const settings = {
    infinite: false,
    speed: 500,
    arrows: true,
    slidesToShow: 4,
    slidesToScroll: 1,
    prevArrow: <FontAwesomeIcon icon={faArrowLeftLong} />,
    nextArrow: <FontAwesomeIcon icon={faArrowRightLong} />,
    responsive: [
      { breakpoint: 1440, settings: { slidesToShow: 3 } },
      { breakpoint: 767, settings: { slidesToShow: 2 } },
      { breakpoint: 539, settings: { slidesToShow: 1 } },
    ],
  };

  useEffect(() => {
    dispatch(fetchEnrolledCourses());
  }, [dispatch]);

  useEffect(() => {
    if (
      enrolledCourses?.courses?.length > 0 &&
      !enrolledCoursesLoader &&
      !isCertificatesLoading &&
      !hasGenerated
    ) {
      console.log("Checking for completed courses to generate certificates...");
      console.log("Certificates data:", certificates);
      generateCertificatesForCompletedCourses();
      setHasGenerated(true);
    }
  }, [enrolledCourses, enrolledCoursesLoader, isCertificatesLoading, hasGenerated, certificates]);

  const downloadImage = async (url, filename = "certificate.png") => {
    console.log("Download URL:", url);
    try {
      const res = await fetch(url);
      if (!res.ok) throw new Error(`Failed to fetch image: ${res.statusText}`);
      const blob = await res.blob();
      const link = document.createElement("a");
      link.href = window.URL.createObjectURL(blob);
      link.download = filename;
      link.click();
      window.URL.revokeObjectURL(link.href);
      toast.success("Certificate downloaded successfully!");
    } catch (error) {
      console.error("Download failed:", error);
      toast.error(`Failed to download certificate: ${error.message}`);
    }
  };

  const generateCertificatesForCompletedCourses = async () => {
    const completedCourses = enrolledCourses.courses?.filter(
      (course) => course.progress_percentage === 100
    ) || [];
    console.log(`Found ${completedCourses.length} completed courses:`, completedCourses.map((c) => c.title));

    for (const course of completedCourses) {
      console.log("Course object:", course);
      const existingCert = certificates.find(
        (cert) => String(cert.course_id) === String(course.id)
      );
      console.log(`Checking certificate for course ID ${course.id} (${course.title}): ${existingCert ? "Already exists" : "Not found, generating..."}`);
      console.log("Existing certificate:", existingCert);

      if (!existingCert && !generatedCerts.includes(course.id)) {
        console.log(`Starting certificate generation for course ID ${course.id} (${course.title})`);
        await generateAndSendCertificate(course);
        setGeneratedCerts((prev) => [...prev, course.id]);
      } else if (existingCert) {
        console.log(`Skipping certificate generation for course ID ${course.id} (${course.title}): Certificate already exists`);
      } else if (generatedCerts.includes(course.id)) {
        console.log(`Skipping certificate generation for course ID ${course.id} (${course.title}): Already generated in this session`);
      }
    }
    console.log("Refetching certificates after generation...");
    refetch();
  };

  const generateAndSendCertificate = async (course) => {
    // Stricter user validation
    console.log(`Validating user for course ID ${course.id} (${course.title}):`, course.user);
    if (
      !course ||
      !course.user ||
      typeof course.user !== "object" ||
      !course.user.id ||
      !course.user.name ||
      course.user.id === null ||
      course.user.name === null
    ) {
      console.error(
        `Cannot generate certificate for course ID ${course.id} (${course.title}): User information is missing or invalid`,
        { user: course.user }
      );
      toast.error(`Failed to generate certificate for "${course.title}": Missing user information`);
      return;
    }

    console.log(`Rendering certificate template for course: ${course.title}`);
    const templateRef = document.createElement("div");
    // Align with CertificateComponent size
    templateRef.style.width = "900px";
    templateRef.style.height = "760px";
    document.body.appendChild(templateRef);
    const root = ReactDOM.createRoot(templateRef);

    // Use current date for new certificates
    const completionDate = new Date().toLocaleDateString("en-GB", {
      day: "2-digit",
      month: "2-digit",
      year: "numeric",
    });

    root.render(
      <CertificateComponent
        courseTitle={course.title}
        studentName={course.user.name}
        completionDate={completionDate}
      />
    );

    await new Promise((resolve) => setTimeout(resolve, 100));

    console.log(`Inspecting DOM for course: ${course.title}`);
    console.log("DOM content:", templateRef.innerHTML);

    console.log(`Generating canvas for certificate of course: ${course.title}`);
    try {
      const canvas = await html2canvas(templateRef, { scale: 2 });
      console.log(`Generated canvas size: ${canvas.width}x${canvas.height}px`);

      if (!canvas || canvas.width === 0 || canvas.height === 0) {
        console.error(`Failed to generate canvas for course ID ${course.id} (${course.title})`);
        toast.error(`Failed to generate certificate for "${course.title}": Canvas generation failed`);
        root.unmount();
        document.body.removeChild(templateRef);
        return;
      }

      const imageData = canvas.toDataURL("image/png");
      console.log(`Certificate image generated for course: ${course.title}`);

      try {
        console.log(`Sending certificate to backend for course ID ${course.id}...`);
        const formData = new FormData();
        formData.append("course_id", course.id);
        formData.append("user_id", course.user.id);

        // Convert base64 to Blob
        const byteString = atob(imageData.split(",")[1]);
        const mimeString = imageData.split(",")[0].split(":")[1].split(";")[0];
        const ab = new ArrayBuffer(byteString.length);
        const ia = new Uint8Array(ab);
        for (let i = 0; i < byteString.length; i++) {
          ia[i] = byteString.charCodeAt(i);
        }
        const blob = new Blob([ab], { type: mimeString });
        formData.append("file_url", blob, `${course.title}-certificate.png`);

        // Log FormData entries
        for (let [key, value] of formData.entries()) {
          console.log(`FormData: ${key} =`, value);
        }

        const response = await addCertificate({ data: formData, token }).unwrap();
        console.log(`Certificate successfully sent to backend for course ID ${course.id}. Response:`, response);
        toast.success(`Certificate for "${course.title}" generated and sent!`);
      } catch (error) {
        console.error(`Error sending certificate for course ID ${course.id}:`, error);
        console.error("Error details:", error.data, error.status);
        const errorMessage =
          error.status === "FETCH_ERROR"
            ? "Failed to send certificate due to CORS or network issue. Please check server configuration."
            : error.data?.message || error.message || "Unknown error occurred";
        toast.error(`Failed to generate certificate for "${course.title}": ${errorMessage}`);
      }
    } catch (error) {
      console.error(`Error generating canvas for course ID ${course.id} (${course.title}):`, error);
      toast.error(`Failed to generate certificate for "${course.title}": ${error.message}`);
    } finally {
      console.log(`Cleaning up certificate template for course: ${course.title}`);
      root.unmount();
      document.body.removeChild(templateRef);
    }
  };

  return (
    <div className="md:px-[28px] px-3 pb-8 min-h-screen">
      <Header title="Certifications" />
      <div className="bg-white mt-6 rounded-2xl w-full max-h-full p-6 mb-4">
        <div className="mb-4">
          <p className="text-[22px] font-bold">Courses Progress</p>
        </div>
        {enrolledCourses?.courses?.length > 0 ? (
          enrolledCoursesLoader ? (
            <div className="pr-2 h-[280px] flex justify-center items-center">
              <LoaderCircle />
            </div>
          ) : (
            <div>
              {enrolledCourses === null ? (
                <div className="flex justify-center items-center min-h-[377px] min-w-full">
                  <LoaderCircle />
                </div>
              ) : enrolledCourses.courses.length > 0 ? (
                <Slider slideCount={enrolledCourses.courses.length} {...settings} className="course-progress-slider">
                  {enrolledCourses.courses.map((course, index) => (
                    <div key={index}>
                      <div className="border border-[#c7c7c7] rounded-xl overflow-hidden">
                        <div className="relative">
                          <Image
                            src={course?.media?.[0]?.url || certificate1}
                            alt={`${course.title} image`}
                            className="h-[220px] w-full object-cover"
                            width={400}
                            height={220}
                            priority
                          />
                          <p
                            className={
                              course?.status === "completed"
                                ? "border-[#0EAD71] text-[#0EAD71] text-[14px] font-medium absolute top-2 right-2 border rounded-full px-2 py-1 bg-white"
                                : "border-[#00CED1] text-[#00CED1] text-[14px] font-medium absolute top-2 right-2 border rounded-full px-2 py-1 bg-white"
                            }
                          >
                            {course?.status === "completed" ? "Done" : "Ongoing"}
                          </p>
                        </div>
                        <div className="p-4">
                          <h3 className="text-[18px] font-bold mb-2">{course?.title || "Untitled Course"}</h3>
                          <div className="flex items-center gap-2 mb-3">
                            <div className="w-full h-[6px] bg-[#D9D9D9] rounded-3xl">
                              <div
                                className="bg-golden rounded-3xl h-[6px]"
                                style={{ width: `${course?.progress_percentage || 0}%` }}
                              ></div>
                            </div>
                            <span>{`${course?.progress_percentage || 0}%`}</span>
                          </div>
                          <div className="flex justify-between items-center">
                            <div className="flex items-center gap-2">
                              <Image
                                src={course?.user?.profile_photo_path || studentProfile}
                                alt="student-profile"
                                className="w-[30px] h-[30px] object-cover rounded-full"
                                width={30}
                                height={30}
                              />
                              <span className="text-[12px]">{course?.user?.name || "Unknown Student"}</span>
                            </div>
                            <Link
                              href={`/courses/${course.id}`}
                              onClick={() => setLoading(true)}
                              className="bg-golden text-white px-4 py-2 rounded-md inline-flex items-center gap-2 text-sm"
                            >
                              {loading ? "Loading..." : "Watch"}
                            </Link>
                          </div>
                        </div>
                      </div>
                    </div>
                  ))}
                </Slider>
              ) : (
                <div className="flex justify-center items-center min-h-[377px] min-w-full">
                  No Courses Available
                </div>
              )}
            </div>
          )
        ) : (
          <div className="p-4 mb-4 text-sm text-blue-800 rounded-lg bg-blue-50 dark:bg-gray-800 dark:text-blue-400 w-full lg:col-span-4 md:col-span-3 sm:col-span-2 col-span-1">
            No Enrolled Course
          </div>
        )}
      </div>
      <div className="bg-white rounded-2xl w-full max-h-full p-6">
        <div className="flex justify-between items-center mb-8">
          <p className="text-[20px] font-bold">Here Are Your Certifications</p>
          <Link href="/certifications/all" className="text-[18px] font-bold">
            See all
          </Link>
        </div>
        <div className="mt-4 grid lg:grid-cols-4 md:grid-cols-3 sm:grid-cols-2 grid-cols-1 min-h-[314px] xl:gap-x-10 gap-x-4 gap-y-4 place-items-center">
          {isCertificatesLoading ? (
            <div className="pr-2 h-[280px] flex justify-center items-center">
              <LoaderCircle />
            </div>
          ) : certificates.length > 0 ? (
            certificates.map((certificate, index) => (
              <div className="w-full" key={index}>
                <img
                  src={certificate?.license || certificateErrorImg.src}
                  alt={`${certificate.course_name} certificate`}
                  className="w-full rounded-lg"
                  width={400}
                  height={300}
                  onError={(e) => {
                    e.currentTarget.src = certificateErrorImg.src;
                  }}
                />
                <div className="flex justify-between mt-3">
                  <p className="text-[#363636] font-bold">{certificate?.course_name || "Unknown Course"}</p>
                  <button
                    onClick={() => downloadImage(certificate?.license || certificateErrorImg.src, `${certificate.course_name}-certificate.png`)}
                    className="text-blue-800 hover:text-blue-600"
                    aria-label={`Download certificate for ${certificate.course_name}`}
                  >
                    <FontAwesomeIcon icon={faDownload} />
                  </button>
                </div>
              </div>
            ))
          ) : (
            <div className="p-4 mb-4 text-sm text-blue-800 rounded-lg bg-blue-50 dark:bg-gray-800 dark:text-blue-400 w-full lg:col-span-4 md:col-span-3 sm:col-span-2 col-span-1">
              No certificate found
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default Certifications;