"use client";
import Image from "next/image";
import React, { useRef, useState } from "react";
import { useFormik } from "formik";
import * as Yup from "yup";
import bgimg from "../../../../public/assets/login/bgimg.png";
import image from "../../../../public/assets/foget-password/image.png";
import Link from "next/link";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faEye, faEyeSlash, faTimes } from "@fortawesome/free-solid-svg-icons";
import toast from "react-hot-toast";
import { useRouter } from "next/navigation";
import Cookies from "js-cookie";
import { useRegisterUserMutation } from "@/services/Api";
import { useDispatch } from "react-redux";
import { loggedIn } from "@/redux/authSlice";

const Signup = () => {
  const router = useRouter();
  const [passwordVisible, setPasswordVisible] = useState(false);
  const [confirmVisible, setConfirmVisible] = useState(false);
  const [error, setError] = useState("");
  const [frontImagePreview, setFrontImagePreview] = useState(null); // State for front ID image preview
  const [frontFileName, setFrontFileName] = useState(null); // State for front ID PDF file name
  const [backImagePreview, setBackImagePreview] = useState(null); // State for back ID image preview
  const [backFileName, setBackFileName] = useState(null); // State for back ID PDF file name
  const frontFileInputRef = useRef(null); // Ref for front ID file input
  const backFileInputRef = useRef(null); // Ref for back ID file input

  const dispatch = useDispatch();
  const [registerUser] = useRegisterUserMutation();

  // ---- File Upload Logic ----
  const handleFileChange = (file, type) => {
    if (file) {
      const allowedTypes = ["image/jpeg", "image/png", "application/pdf"];
      if (allowedTypes.includes(file.type)) {
        if (type === "front") {
          setFrontFileName(file.name);
          if (file.type.startsWith("image/")) {
            setFrontImagePreview(URL.createObjectURL(file));
          } else {
            setFrontImagePreview(null);
          }
          formik.setFieldValue("id_card_front", file);
        } else {
          setBackFileName(file.name);
          if (file.type.startsWith("image/")) {
            setBackImagePreview(URL.createObjectURL(file));
          } else {
            setBackImagePreview(null);
          }
          formik.setFieldValue("id_card_back", file);
        }
        // toast.success(`File "${file.name}" selected successfully!`, {
        //   position: "top-center",
        // });
      } else {
        toast.error("Please upload a valid JPG, PNG, or PDF file.", {
          position: "top-center",
        });
      }
    }
  };

  const handleDrop = (event, type) => {
    event.preventDefault();
    event.stopPropagation();
    const file = event.dataTransfer.files[0];
    handleFileChange(file, type);
  };

  const handleFileInput = (event, type) => {
    const file = event.target.files[0];
    handleFileChange(file, type);
  };

  const handleDragOver = (event) => {
    event.preventDefault();
    event.stopPropagation();
  };

  const removeFile = (type) => {
    if (type === "front") {
      setFrontImagePreview(null);
      setFrontFileName(null);
      formik.setFieldValue("id_card_front", null);
      if (frontFileInputRef.current) frontFileInputRef.current.value = "";
    } else {
      setBackImagePreview(null);
      setBackFileName(null);
      formik.setFieldValue("id_card_back", null);
      if (backFileInputRef.current) backFileInputRef.current.value = "";
    }
  };

  // ---- Formik Logic ----
  const formik = useFormik({
    initialValues: {
      name: "",
      email: "",
      address: "",
      city: "",
      state: "",
      zip_code: "",
      password: "",
      password_confirmation: "",
      id_card_front: null,
      id_card_back: null,
    },

    validationSchema: Yup.object({
      name: Yup.string().min(2).max(50).required("Name is required"),
      email: Yup.string().email("Invalid email").required("Email is required"),
      address: Yup.string().required("Addres is required"),
      city: Yup.string().required("City is required"),
      state: Yup.string().required("State is required"),
      zip_code: Yup.string().required("Zip code is required"),
      password: Yup.string()
        .required("Password is required")
        .min(8, "At least 8 characters")
        .matches(/[a-zA-Z]/, "Only Latin letters allowed")
        .matches(/[!@#$%^&*(),.?":{}|<>]/, "Must contain a special character"),
      password_confirmation: Yup.string()
        .oneOf([Yup.ref("password"), null], "Passwords must match")
        .required("Confirm Password is required"),
      id_card_front: Yup.mixed().required("Front ID file is required"),
      id_card_back: Yup.mixed().required("Back ID file is required"),
    }),

    onSubmit: async (values, { setSubmitting }) => {
      try {
        const formData = new FormData();
        formData.append("name", values.name);
        formData.append("email", values.email);
        formData.append("password", values.password);
        formData.append("address", values.address);
        formData.append("city", values.city);
        formData.append("state", values.state);
        formData.append("zip_code", values.zip_code);
        formData.append("password_confirmation", values.password_confirmation);
        if (values.id_card_front) {
          formData.append("id_card_front", values.id_card_front);
        }
        if (values.id_card_back) {
          formData.append("id_card_back", values.id_card_back);
        }

        const res = await registerUser({ data: formData }).unwrap();

        // Agar success mila
        const token = res?.token;
        const role = res?.roles[0];
        localStorage.setItem("authToken", token);

        const userDetail = {
          id: res?.user?.id,
          name: res?.user?.name,
          role: res?.user?.roles[0],
        };

        dispatch(
          loggedIn({
            token,
            user: userDetail,
            user_verified: res?.user_verified,
          })
        );

        Cookies.set("authToken", token, { path: "/otp", expires: 7 });
        Cookies.set("role", role, { path: "/otp", expires: 7 });
        router.push("/otp");
        toast.success("Signup successful!", { position: "top-center" });
      } catch (error) {
        console.log("Signup error:", error);

        if (error?.data?.errors) {
          Object.values(error.data.errors).forEach((errArr) => {
            toast.error(errArr[0], { position: "top-center" });
          });
        } else {
          toast.error("Something went wrong", { position: "top-center" });
        }
      } finally {
        setSubmitting(false);
      }
    },
  });

  return (
    <div className="w-full min-h-full flex justify-between relative">
      <div className="absolute z-10 min-h-full flex items-center left-0 lg:left-[1.5%] sm:max-w-[900px] w-full px-4 px-sm-0">
        <div className="flex flex-col justify-around min-h-full w-full py-4 ">
          <div className="flex justify-center">
            <Image
              src="/assets/navbar/logo1.svg"
              alt="logo"
              className="w-40 h-40 lg:w-52 lg:h-40 mb-6 md:mb-4"
              width={100}
              height={100}
            />
          </div>
          {/* Form */}
          <div className=" bg-whitegrey px-5 py-8 rounded-2xl">
            <div className="flex flex-col gap-3">
              <h1 className="text-black text-3xl font-bold text-center">
                Registration
              </h1>
            </div>
            <form
              className="grid grid-cols-1 md:grid-cols-2 md:gap-4 gap-4"
              onSubmit={formik.handleSubmit}
            >
              {/* Name */}
              <div className="flex flex-col gap-2">
                <label>Name:</label>
                <input
                  type="text"
                  name="name"
                  value={formik.values.name}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  placeholder="Enter your Name"
                  className={`outline-none py-2 px-3 border rounded-xl ${
                    formik.touched.name && formik.errors.name
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                />
                {formik.touched.name && formik.errors.name && (
                  <span className="text-red-500">{formik.errors.name}</span>
                )}
              </div>

              {/* Email */}
              <div className="flex flex-col gap-2">
                <label>Email:</label>
                <input
                  type="email"
                  name="email"
                  value={formik.values.email}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  placeholder="Enter your Email"
                  className={`outline-none py-2 px-3 border rounded-xl ${
                    formik.touched.email && formik.errors.email
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                />
                {formik.touched.email && formik.errors.email && (
                  <span className="text-red-500">{formik.errors.email}</span>
                )}
              </div>

              <div className="flex flex-col gap-2">
                <label>Address:</label>
                <input
                  type="text"
                  name="address"
                  value={formik.values.address}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  placeholder="Enter your address"
                  className={`outline-none py-2 px-3 border rounded-xl ${
                    formik.touched.address && formik.errors.address
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                />
                {formik.touched.address && formik.errors.address && (
                  <span className="text-red-500">{formik.errors.address}</span>
                )}
              </div>

              <div className="flex flex-col gap-2">
                <label>City:</label>
                <input
                  type="text"
                  name="city"
                  value={formik.values.city}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  placeholder="Enter your city"
                  className={`outline-none py-2 px-3 border rounded-xl ${
                    formik.touched.city && formik.errors.city
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                />
                {formik.touched.city && formik.errors.city && (
                  <span className="text-red-500">{formik.errors.city}</span>
                )}
              </div>

              <div className="flex flex-col gap-2">
                <label>State:</label>
                <input
                  type="text"
                  name="state"
                  value={formik.values.state}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  placeholder="Enter your state"
                  className={`outline-none py-2 px-3 border rounded-xl ${
                    formik.touched.state && formik.errors.state
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                />
                {formik.touched.state && formik.errors.state && (
                  <span className="text-red-500">{formik.errors.state}</span>
                )}
              </div>

              <div className="flex flex-col gap-2">
                <label>Zip Code:</label>
                <input
                  type="text"
                  name="zip_code"
                  value={formik.values.zip_code}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  placeholder="Enter your zip code"
                  className={`outline-none py-2 px-3 border rounded-xl ${
                    formik.touched.zip_code && formik.errors.zip_code
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                />
                {formik.touched.zip_code && formik.errors.zip_code && (
                  <span className="text-red-500">{formik.errors.zip_code}</span>
                )}
              </div>

              {/* Front ID Upload */}
              <div className="flex flex-col gap-2">
                <label>
                  Upload Front of ID:{" "}
                  <span className="opacity-50 text-sm">(JPG, PNG, or PDF)</span>
                </label>
                <div
                  className="border-2 border-dashed rounded-md h-[150px] flex flex-col items-center justify-center text-sm cursor-pointer bg-gray-300 px-3"
                  onDragOver={handleDragOver}
                  onDrop={(e) => handleDrop(e, "front")}
                  onClick={() => frontFileInputRef.current.click()}
                >
                  <p>Drag & drop a file here or click to upload</p>
                  <input
                    type="file"
                    name="id_card_front"
                    accept="image/jpeg,image/png,application/pdf"
                    className="hidden"
                    ref={frontFileInputRef}
                    onChange={(e) => handleFileInput(e, "front")}
                  />
                </div>
                {frontImagePreview && (
                  <div className="mt-2 flex items-center gap-3">
                    <Image
                      src={frontImagePreview}
                      alt="Front ID Preview"
                      width={200}
                      height={150}
                      className="object-cover rounded-md"
                    />
                    <button
                      type="button"
                      className="text-red-500 font-bold text-sm hover:underline"
                      onClick={() => removeFile("front")}
                    >
                      Delete
                    </button>
                  </div>
                )}
                {frontFileName && !frontImagePreview && (
                  <div className="mt-2 relative inline-block bg-gray-100 p-2 rounded-md">
                    <span>{frontFileName}</span>
                    <button
                      type="button"
                      className="ml-2 text-red-600 font-bold"
                      onClick={() => removeFile("front")}
                    >
                      ✕
                    </button>
                  </div>
                )}
                {formik.touched.id_card_front &&
                  formik.errors.id_card_front && (
                    <span className="text-red-500">
                      {formik.errors.id_card_front}
                    </span>
                  )}
              </div>

              {/* Back ID Upload */}
              <div className="flex flex-col gap-2">
                <label>
                  Upload Back of ID:{" "}
                  <span className="opacity-50 text-sm">(JPG, PNG, or PDF)</span>
                </label>
                <div
                  className="border-2 border-dashed rounded-md h-[150px] flex flex-col items-center justify-center text-sm cursor-pointer bg-gray-300 px-3"
                  onDragOver={handleDragOver}
                  onDrop={(e) => handleDrop(e, "back")}
                  onClick={() => backFileInputRef.current.click()}
                >
                  <p>Drag & drop a file here or click to upload</p>
                  <input
                    type="file"
                    name="id_card_back"
                    accept="image/jpeg,image/png,application/pdf"
                    className="hidden"
                    ref={backFileInputRef}
                    onChange={(e) => handleFileInput(e, "back")}
                  />
                </div>
                {backImagePreview && (
                  <div className="mt-2 flex items-center gap-3">
                    <Image
                      src={backImagePreview}
                      alt="Back ID Preview"
                      width={200}
                      height={150}
                      className="object-cover rounded-md"
                    />
                    <button
                      type="button"
                      className="text-red-500 font-bold text-sm hover:underline"
                      onClick={() => removeFile("back")}
                    >
                      Delete
                    </button>
                  </div>
                )}
                {backFileName && !backImagePreview && (
                  <div className="mt-2 relative inline-block bg-gray-100 p-2 rounded-md">
                    <span>{backFileName}</span>
                    <button
                      type="button"
                      className="ml-2 text-red-600 font-bold"
                      onClick={() => removeFile("back")}
                    >
                      ✕
                    </button>
                  </div>
                )}
                {formik.touched.id_card_back && formik.errors.id_card_back && (
                  <span className="text-red-500">
                    {formik.errors.id_card_back}
                  </span>
                )}
              </div>

              {/* Passwords */}
              <div className="relative flex flex-col gap-2">
                <label>Password:</label>
                <input
                  type={passwordVisible ? "text" : "password"}
                  className={`outline-none py-2 px-3 border border-grey rounded-xl ${
                    formik.touched.password && formik.errors.password
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                  placeholder="Enter New Password"
                  name="password"
                  value={formik.values.password}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                />
                <button
                  type="button"
                  onClick={() => setPasswordVisible(!passwordVisible)}
                  className="absolute text-darkgrey right-3 top-10"
                >
                  {passwordVisible ? (
                    <FontAwesomeIcon icon={faEyeSlash} />
                  ) : (
                    <FontAwesomeIcon icon={faEye} />
                  )}
                </button>
                {formik.touched.password && formik.errors.password ? (
                  <div className="text-red-500">{formik.errors.password}</div>
                ) : null}
              </div>
              <div className="relative flex flex-col gap-2">
                <label>Confirm Password:</label>
                <input
                  type={confirmVisible ? "text" : "password"}
                  className={`outline-none py-2 pr-10 px-3 border border-grey rounded-xl ${
                    formik.touched.password_confirmation &&
                    formik.errors.password_confirmation
                      ? "border-red-500"
                      : "border-grey"
                  }`}
                  placeholder="Enter Retype New Password"
                  name="password_confirmation"
                  value={formik.values.password_confirmation}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                />
                <button
                  type="button"
                  onClick={() => setConfirmVisible(!confirmVisible)}
                  className="absolute text-darkgrey right-3 top-10"
                >
                  {confirmVisible ? (
                    <FontAwesomeIcon icon={faEyeSlash} />
                  ) : (
                    <FontAwesomeIcon icon={faEye} />
                  )}
                </button>
                {formik.touched.password_confirmation &&
                formik.errors.password_confirmation ? (
                  <div className="text-red-500">
                    {formik.errors.password_confirmation}
                  </div>
                ) : null}
              </div>
              {error && <div className="text-red-500 text-center">{error}</div>}
              <button
                type="submit"
                disabled={formik.isSubmitting}
                className="bg-golden mt-10 text-white py-2 px-3 rounded-xl md:col-span-2"
              >
                {formik.isSubmitting ? "Signing up..." : "Sign Up"}
              </button>
            </form>
          </div>
        </div>
      </div>

      <div></div>

      {/* Right Side */}
      <div className="min-h-screen relative flex justify-end w-full md:w-2/3 lg:w-1/2">
        <Image src={bgimg} alt="bgimg" className="z-0 w-full min-h-full" />
        <div className="absolute hidden md:flex justify-center items-center h-full w-full">
          <Image
            src={image}
            alt="bgimg"
            className="lg:min-h-[360px] min-h-[300px] xl:max-w-[500px] lg:max-w-[300px] md:max-w-[300px]"
          />
        </div>
      </div>
    </div>
  );
};

export default Signup;
